/*
 * Protocol independent interface for GM
 */
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <pthread.h>
#include <ctype.h>
#include <poll.h>
#include <ctype.h>
#include <sys/socket.h>
#include <netinet/tcp.h>
#include <netdb.h>
#include <netinet/in.h>
#include <signal.h>
#include <arpa/inet.h>

#include "libfma.h"
#include "libmyri.h"
#include "lf_myri_packet.h"


typedef unsigned sm_u32_t;
typedef unsigned char sm_u8_t;

#define SM_SOCKET_NAME_LENGTH 182

#define LF_ERROR(e) do {						\
  printf ("*** %s:%d:%s() ", __FILE__, __LINE__, __FUNCTION__);		\
  printf e;								\
  printf(" ***\nerrno=%d(%s)\n", errno, strerror (errno));		\
} while (0)

enum
{
  SM_SOCKET_CONNECT,
  SM_SOCKET_DATA,
  SM_SOCKET_STOP,
  SM_SOCKET_GO,
  SM_SOCKET_TAIL,
  SM_SOCKET_BEAT,
  SM_SOCKET_ILLEGAL,
  SM_SOCKET_DLLP,
  SM_SOCKET_TLP,
  SM_SOCKET_MAC_ADDRESS
};

typedef struct sm_socket_packet_t
{
  sm_u32_t type;
  sm_u32_t host;
  sm_u32_t data;
  
} sm_socket_packet_t;

typedef struct sm_socket_mac_address_t
{
  sm_u32_t type;
  sm_u8_t mac_address [6];
}sm_socket_mac_address_t;

struct lxgdb_t;
struct raw_send_t;

typedef struct raw_send_t {
  int port;
  char buffer [LF_MAX_MYRI_RAW_PKT_LEN + MYRI_MAX_ROUTE_LEN + 5];
  int length;
  void *context;
  int stopped;
  
  struct lxgdb_t *mip;

  struct raw_send_t*next;
  struct raw_send_t*prev;
} raw_send_t;

typedef struct lxgdb_t
{
  int rd_fd;
  int wr_fd;
  int data_fd;
  int flow_fd;
  pthread_mutex_t event_queue_lock;
  pthread_mutex_t send_queue_lock;
  struct myri_event event_queue;
  struct myri_event null_event;		/* used for returning NO_EVENT */
  pthread_t raw_thread;
  int bytes_received;
  char receive_buffer [LF_MAX_MYRI_RAW_PKT_LEN + MYRI_MAX_ROUTE_LEN];
  raw_send_t send_queue;
  raw_send_t*pending_send;
  int done;
  int stopped;
  int bytes_sent;
  int rawpipe [2];
  int port;
}lxgdb_t;

static lxgdb_t lxgdb;

static int something_to_send (lxgdb_t*mip)
{
  return mip->send_queue.prev != &mip->send_queue || mip->pending_send;
}

static int read_socket (int fd, void*_buffer, int length)
{
  int total, n;
  char *buffer = _buffer;

  for (total = 0; total < length; total += n)
  {
    n = read (fd, buffer + total, length - total);
    if (n == -1) {
      if (errno != EINTR
#if HAVE_DECL_ERESTART
	  && errno != ERESTART
#endif
	  && errno != EAGAIN) {
	return 0;
      }
      printf ("%s(): ignoring %d: %s", __FUNCTION__, errno, strerror (errno));
      n = 0;
      continue;
    }
    else if (n == 0) {
      errno = 0;
      return 0;
    }
  }
  return total;
}

static int write_socket (int fd, char*buffer, int length)
{
  int total, n;

  for (total = 0; total < length; total += n)
  {
    n = write (fd, buffer + total, length - total);
    if (n == -1) {
      if (errno != EINTR
#if HAVE_DECL_ERESTART
	  && errno != ERESTART
#endif
	  && errno != EAGAIN) {
	return 0;
      }
      n = 0;
      continue;
    }
    else if (n == 0) {
      errno = 0;
      return 0;
    }
  }
  return total;
}


int lxgdb_connect (lxgdb_t*mip, char*server, int server_port, int host, int port)
{
  struct hostent *he;
  struct sockaddr_in sin;
  unsigned int sin_addr;
  int option;
  sm_socket_packet_t p;
  sm_socket_mac_address_t mp;
  
  /*insist (server_port > 0);*/
  
  if ((he = gethostbyname (server)))
    memcpy ((char*)&sin_addr, he->h_addr, he->h_length);
  else if ((sin_addr = inet_addr (server)) == (unsigned) -1)
  {
    LF_ERROR (("can't get IP address of %s", server));
    return 0;
  }

  /* Connect to the server. */
  
  memset (&sin, 0, sizeof (sin));
  sin.sin_family = AF_INET;
  sin.sin_port = htons (server_port);
  sin.sin_addr.s_addr = sin_addr;
  
  if ((mip->data_fd = socket (PF_INET, SOCK_STREAM, 0)) < 0)
    LF_ERROR (("socket failed"));

  if (connect (mip->data_fd, (struct sockaddr*) &sin, sizeof (sin)) < 0)
  {
    perror ("connect failed");
    return 0;
  }

  option = 1;
  setsockopt (mip->data_fd, SOL_SOCKET, TCP_NODELAY, (char*) &option, sizeof (option));

  option = sizeof (sm_socket_packet_t) * 1000;
  setsockopt (mip->data_fd, SOL_SOCKET, SO_SNDBUF, (char*) &option, sizeof (option));

  option = sizeof (sm_socket_packet_t) * 1000;
  setsockopt (mip->data_fd, SOL_SOCKET, SO_RCVBUF, (char*) &option, sizeof (option));
  
  p.type = htonl (SM_SOCKET_CONNECT);
  p.host = htonl (host);
  p.data = htonl (port);
  if (write (mip->data_fd, (char*) &p, sizeof (p)) != sizeof (p)) {
    LF_ERROR (("first write failed"));
    return 0;
  }
  
  if (read (mip->data_fd, (char*) &mp, sizeof (mp)) != sizeof (mp)) {
    LF_ERROR (("first_read failed"));
    return 0;
  }
  
  /*insist (htonl (mp.type) == SM_SOCKET_MAC_ADDRESS);*/

  /*for (i = 0; i < 6; i++)
    mac_address [i] = mp.mac_address [i];  
  */

  /* Get the flow control TCP port from the server, and connect to it. */

  if (read_socket (mip->data_fd, &sin.sin_port, sizeof (sin.sin_port))
      != sizeof (sin.sin_port)) {
    perror ("flow control port read failed");
    return 0;
  }
  
  if ((mip->flow_fd = socket (PF_INET, SOCK_STREAM, 0)) < 0)
  {
    perror ("socket failed");
    return 0;
  }
  

  if (connect (mip->flow_fd, (struct sockaddr*) &sin, sizeof (sin)) < 0)
  {
    perror ("connect failed");
    return 0;
  }
  return 1;
}

/*
 * Enqueue an event and let the main thread know about it
 */
static void enqueue_event(
  struct lxgdb_t *mip,
  struct myri_event *mep)
{
  char junk;
  int rc;

  /* lock while we manipulate the queue */
  pthread_mutex_lock(&mip->event_queue_lock);

  /* link this element into the head of the list */
  mep->next = mip->event_queue.next;
  mep->prev = &mip->event_queue;

  mep->next->prev = mep;
  mep->prev->next = mep;

  /* all done with the queue */
  pthread_mutex_unlock(&mip->event_queue_lock);

  /* write a byte to the pipe to wake up main thread */
  rc = send(mip->wr_fd, &junk, 1, 0);
  if (rc != 1) {
    perror("Writing to pipe");
    exit(1);
  }
}

static int myri_open_sockets(
  lxgdb_t*mip)
{
#if 1
  int sv[2];
  
  if (socketpair (PF_UNIX, SOCK_STREAM, 0, sv) != 0)
    return -1;
  mip->rd_fd = sv[0];
  mip->wr_fd = sv[1];
  return 0;
#else
  int sock;
  struct sockaddr_in addr;
  socklen_t len;
  int rc;

  sock = socket(PF_INET, SOCK_STREAM, 0);
  if (sock == -1) {
    perror("opening socket");
    goto abort_with_nothing;
  }
  addr.sin_family = AF_INET;
  addr.sin_port = 0;
  addr.sin_addr.s_addr = INADDR_ANY;
  rc = bind(sock, (struct sockaddr*)&addr, sizeof(addr));
  if (rc == -1) {
    perror("binding socket");
    goto abort_with_sock;
  }
  rc = listen(sock, 1);
  if (rc == -1) {
    perror("listening");
    goto abort_with_sock;
  }
  mip->wr_fd = socket(PF_INET, SOCK_STREAM, 0);
  if (mip->wr_fd == -1) {
    perror("opening writer");
    goto abort_with_sock;
  }
  len = sizeof(addr);
  rc = getsockname(sock, (struct sockaddr*)&addr, &len);
  if (rc == -1) {
    perror("getting sock name");
    goto abort_with_writer;
  }
  rc = connect(mip->wr_fd, (struct sockaddr*)&addr, sizeof(addr));
  if (rc == -1) {
    perror("connecting writer");
    goto abort_with_writer;
  }
  mip->rd_fd = accept(sock, NULL, NULL);
  if (mip->rd_fd == -1) {
    perror("connecting reader");
    goto abort_with_writer;
  }
  close(sock);
  return 0;

 abort_with_writer:
  close(mip->wr_fd);
 abort_with_sock:
  close(sock);
 abort_with_nothing:
  return -1;
#endif
}

#ifdef PRINT
static void print_buffer (char*p, int length)
{
  int i;
  char b1 [64];
  char b2 [64];
  char*s1,*s2;
  
  s1 = b1;
  s2 = b2;
  
  for (i = 0; i < length; i++)
  {
    s1 += sprintf (s1, "%2.2x ", p [i] & 0xff);
    s2 += sprintf (s2, "%c", isalnum (p [i]) ? p[i] : '.');
    
    if ((i % 16) == 15)
    {
      s1 = b1;
      s2 = b2;
      printf ("%s    %s\n", s1, s2);
    }
  }
  if (s1 != b1)
    printf ("%s    %s\n", s1, s2);
}
#endif

/*
 * raw communication thread
 */
static void *
lxgdb_raw_thread(
  void *arg)
{
  sm_socket_packet_t p;
  fd_set rd_fds, wr_fds;
  lxgdb_t *mip;
  struct myri_event *mep;
  void *rx_buf;
  int highest_fd;
  char junk;  

  mip = arg;
  rx_buf = NULL;
  mep = NULL;
  
  /*
   * service our sockets and generate events for the main thread*/

  highest_fd = 0;
  if (mip->flow_fd > highest_fd)
    highest_fd = mip->flow_fd;
  if (mip->data_fd > highest_fd)
    highest_fd = mip->data_fd;
  if (mip->rawpipe[0] > highest_fd)
    highest_fd = mip->rawpipe[0];

  while (mip->done == 0) {

    FD_ZERO (&rd_fds);
    FD_ZERO (&wr_fds);
    FD_SET (mip->flow_fd, &rd_fds);
    FD_SET (mip->data_fd, &rd_fds);
    FD_SET (mip->rawpipe[0], &rd_fds);
    
    if (something_to_send (mip))
    {
      //printf ("something to send\n");
      FD_SET (mip->data_fd, &wr_fds);
    }
    
    if (select (highest_fd + 1, &rd_fds, &wr_fds, 0, 0) > 0)
    {
      //printf ("raw_thread woken up\n");
      if (FD_ISSET (mip->rawpipe[0], &rd_fds))
      {  
	if (read(mip->rawpipe[0], &junk, 1) != 1)
	{
	  LF_ERROR (("recv failed on rawpipe[0]"));
	}
      }
      
      if (FD_ISSET (mip->flow_fd, &rd_fds))
      {
	//printf ("something to read on flow_fd\n");
	/*something arrived on flow control socket*/
	if (read_socket (mip->flow_fd, (char*)&p, sizeof (p)) != sizeof (p))
	{
	  LF_ERROR (("read_socket failed on flow_fd"));
	}
	if (htonl (p.type) == SM_SOCKET_STOP)
	{
	  printf ("stopped\n");
	  mip->stopped = 1;
	}
	else if (htonl (p.type) == SM_SOCKET_GO)
	{
	  printf ("going\n");
	  mip->stopped = 0;
	}
	else
	{
	  LF_ERROR (("bad type on flow_fd"));
	}
      }
      if (FD_ISSET (mip->data_fd, &rd_fds))
      {
	/*something arrived on data socket*/
	//printf ("something to read on data_fd\n");

	if (read_socket (mip->data_fd, (char*)&p, sizeof (p)) != sizeof (p))
	{
	  LF_ERROR (("read_socket failed on data_fd"));
	}
	if (htonl (p.type) == SM_SOCKET_DATA)
	{
	  if (mip->bytes_received >= LF_MAX_MYRI_RAW_PKT_LEN)
	  {
	    LF_ERROR (("received too many bytes"));
	  }
	  else
	    mip->receive_buffer [mip->bytes_received++] = (char) htonl (p.data);
	  
	}
	else if (htonl (p.type) == SM_SOCKET_TAIL)
	{
	  /*end of a packet. queue up recv event*/

	  /* get a struct to hold event */
	  mep = (struct myri_event *) malloc(sizeof(*mep));
	  if (mep == NULL) {
	    perror("allocating raw event struct");
	    exit(1);
	  }

	  /* get a receive buffer*/
	  rx_buf = malloc(LF_MAX_MYRI_RAW_PKT_LEN);
	  if (rx_buf == NULL) {
	    perror("allocating raw receive buffer");
	    exit(1);
	  }

	  printf ("recv packet length %d\n", mip->bytes_received);
	  mep->type = MYRI_EVENT_RAW_RECV_COMPLETE;
	  mep->d.raw_recv.rxbuf = rx_buf;
	  mep->d.raw_recv.rx_len = mip->bytes_received;
	  mep->d.raw_recv.port = mip->port;
	
	  /* copy over the data */
	  memcpy(rx_buf, mip->receive_buffer, mep->d.raw_recv.rx_len);

	  enqueue_event(mip, mep);

	  /* clear receive*/
	  mip->bytes_received = 0;

	}
	else
	{
	  LF_ERROR (("bad type on data_fd"));
	}

      }
      if (FD_ISSET (mip->data_fd, &wr_fds))
      {
	/*we can write data. and we have data to write. write a byte*/
	//printf ("something to write on data_fd\n");
	if (!mip->pending_send)
	{
	  raw_send_t*sp = mip->send_queue.prev;
	  if (sp && sp != &mip->send_queue)
	  {
	    /*we know this has to be true but the fma doesn't like assertions*/
	    sp->prev->next = sp->next;
	    sp->next->prev = sp->prev;
	    mip->pending_send = sp;
	    mip->bytes_sent = 0;
	  }
	}
	
	if (mip->pending_send)
	{
	  /*if the send is done or if the send is out the wrong port*/
	  if (mip->pending_send->port != mip->port || mip->bytes_sent == mip->pending_send->length)
	  {
	    /*only actually send tail if send is out the connected port*/
	    if (mip->pending_send->port == mip->port)
	    {
	      p.type = htonl (SM_SOCKET_TAIL);
	      if (write_socket (mip->data_fd, (char*)&p, sizeof (p))
		  != sizeof (p))
		LF_ERROR (("write failed"));
#ifdef PRINT
	      printf ("sent packet length %d\n", mip->pending_send->length);
	      print_buffer (mip->pending_send->buffer, mip->pending_send->length);
#endif
	    }
	    
	    /*done w/ a packet, queue up a send done event*/
	    /* get a struct to hold event */
	    mep = (struct myri_event *) malloc(sizeof(*mep));
	    if (mep == NULL) {
	      perror("allocating raw event struct");
	      exit(1);
	    } 
	    mep->type = MYRI_EVENT_RAW_SEND_COMPLETE;
	    mep->d.raw_send.context = mip->pending_send->context;
	    enqueue_event(mip, mep);
	    free (mip->pending_send);
	    mip->pending_send = 0;
	  }
	  else
	  {  
	    p.type = htonl (SM_SOCKET_DATA);
	    p.data = htonl (mip->pending_send->buffer [mip->bytes_sent++]);
	    if (write_socket (mip->data_fd, (char*)&p, sizeof (p))
		!= sizeof (p))
	      LF_ERROR (("write failed"));
	  }
	}
      }
    }
  }
  return NULL;
}


/*
 * Instantiate the myri_ interface
 */
int
myri_open(
  int nic_id)
{
  int rc;
  lxgdb_t*mip = &lxgdb;
  char server [255];
  int server_port;
  int host_index, host_port;
  char*s;
  
  if (nic_id != 0) return -1;		/* only one NIC */

  /*get sim server connect info from an env variable*/
  s = getenv ("LXGDB_CONNECT_STRING");
  if (!s)
  {
    LF_ERROR (("LXGDB_CONNECT_STRING not defined"));
    return -1;
  }
  
  /*allow buffer overruns*/
  if (sscanf (s, "%s %d:%d:%d", server, &server_port, &host_index, &host_port) != 4)
  {
    LF_ERROR (("malformed LXGDB_CONNECT_STRING"));
    return -1;
  }

  if (!lxgdb_connect (mip, server, server_port, host_index, host_port))
    return -1;

  mip->rd_fd = -1;
  mip->port = host_port;
  
  /* empty the event queue */
  mip->event_queue.next = &mip->event_queue;
  mip->event_queue.prev = &mip->event_queue;

  /* empty the send queue */
  mip->send_queue.next = &mip->send_queue;
  mip->send_queue.prev = &mip->send_queue;

  /* initialize null event */
  mip->null_event.type = MYRI_EVENT_NO_EVENT;

  /* set up the pipe we will be using for notifying of events */
  rc = myri_open_sockets(mip);
  if (rc == -1) {
    perror("opening pipe");
    return -1;
  }

  /*make pipe to wake raw thread on*/
  rc = pipe (mip->rawpipe);
  if (rc == -1)
  {
    perror ("making raw pipe");
    return -1;
  }
  
  /*reset stuff*/
  mip->bytes_received = 0;
  mip->bytes_sent = 0;
  mip->pending_send = 0;
  mip->stopped = 0;
  mip->done = 0;
  mip->stopped = 0;
  
  /* allocate locks */
  pthread_mutex_init(&mip->event_queue_lock, NULL);
  pthread_mutex_init(&mip->send_queue_lock, NULL);

  /* start the raw thread */
  rc = pthread_create(&mip->raw_thread, NULL, lxgdb_raw_thread, mip);
  if (rc != 0) {
    perror("Spawning raw thread");
    return -1;
  }

  /* we don't really care about the nic id*/
  return 0;
}


/*
 * Close a myri instance
 */
void
myri_close(
  int nic_id)
{
  /*prefectly happy to leak memory*/
}

/*
 * Return the file descriptor associated with a myri handle
 */
int
myri_fd(
  int nic_id)
{
  return lxgdb.rd_fd;
}

/*
 * Get the next event from the raw thread
 */
int
myri_next_event(
  int nic_id,
  struct myri_event **rep,
  int timeout)
{
  lxgdb_t*mip = &lxgdb;
  struct myri_event*mep;
  int rc;
  char junk;
  
  /*
   * If a non-negative timeout is specified, poll on the pipe for
   * that long and return NO_EVENT if timeout occurs
   */
  if (timeout >= 0) {
    struct pollfd pf;

    /* execute a poll() with appropriate timeout */
    pf.fd = mip->rd_fd;
    pf.events = POLLIN;
    rc = poll(&pf, 1, timeout);

    /* bad if error and not EINTR */
    if (rc == -1 && errno != EINTR
#if HAVE_DECL_ERESTART
	&& errno != ERESTART
#endif
	&& errno != EAGAIN) {
      return -1;
    }

    /* return of 0 from poll means nothing is ready */
    if (rc == 0 || (pf.revents & POLLIN) == 0) {
      *rep = &mip->null_event;
      return 0;
    }
  }

  /* read one byte from the pipe to ensure an event is ready */
  rc = recv(mip->rd_fd, &junk, 1, 0);
  if (rc != 1) return -1;

  /* lock while we manipulate the queue */
  pthread_mutex_lock(&mip->event_queue_lock);

  mep = mip->event_queue.prev;
  if (mep == &mip->event_queue) {
    fprintf(stderr, "event queue unexpectedly empty!\n");
    return -1;
  }
  
  /* unlink this element from the list*/
  mep->next->prev = mep->prev;
  mep->prev->next = mep->next;

  /* all done with the queue */

  if (mep->type == MYRI_EVENT_RAW_RECV_COMPLETE)
  {
#ifdef PRINT
    printf ("raw recv:\n");
    print_buffer (mep->d.raw_recv.rxbuf,mep->d.raw_recv.rx_len);
#endif
  } 
  pthread_mutex_unlock(&mip->event_queue_lock);
 
  *rep = mep;
  return 0;
}


/*
 * Release an event struct returned by myri_next_event
 */
void
myri_release_event(
  struct myri_event *mep)
{
  /* NO_EVENT is passed via a static event buffer */
  if (mep->type == MYRI_EVENT_NO_EVENT) {
    return;
  }

  if (mep->type == MYRI_EVENT_RAW_RECV_COMPLETE) {
    free(mep->d.raw_recv.rxbuf);
  }
  free(mep);
}

/*
 * Perform a raw send
 */
int
myri_raw_send(
  int nic_id,
  int port,
  void *route,
  int route_len,
  void *txbuf,
  int txlen,
  void *context)
{
  lxgdb_t*mip;
  raw_send_t*sp;
  int was_sending;
  
  //printf ("raw_send out port %d\n", port);
  
  mip = &lxgdb;
  
  /* allocate a send queue entry */
  sp = (raw_send_t *) calloc(sizeof(*sp), 1);
  if (sp == NULL) {
    perror("Error allocating send queue entry");
    return -1;
  }

  /* fill it in */
  sp->mip = mip;
  sp->length = txlen;
  sp->context = context;
  sp->port = port;
  
  /*copy route and body to bufer*/
  /*assert (route_len > = 0 && txlen > 0 && route_len <= MAX_ROUTE_LEN && txlen <= LF_MAX_MYRI_RAW_PACKET_LEN);*/
  memcpy (sp->buffer, route, route_len);
  memcpy (sp->buffer + route_len, txbuf, txlen);

  /*add 5 bytes for a crc that we don't care about*/
  sp->length += 5;
  
  /*add to send queue for the raw thread*/
  pthread_mutex_lock(&mip->send_queue_lock);
  was_sending = something_to_send (mip);
  
  sp->next = mip->send_queue.next;
  sp->prev = &mip->send_queue;
  sp->next->prev = sp;
  sp->prev->next = sp;

  /*if sending had been idle, make sure our select call
    unblocks now that we have something to send*/

  if (!was_sending)
  {
    char junk;
    
    //printf ("waking up raw thread\n");
    // pthread_kill (mip->raw_thread, SIGURG);

    /* write a byte to the pipe to wake up raw thread */
    if (write (mip->rawpipe[1], &junk, 1) != 1)
    {
      perror("Writing to rawpipe");
      exit(1);
    }
    
  }
  

  pthread_mutex_unlock(&mip->send_queue_lock);
  return 0;
}

/*
 * Get the hostname of a remote host.  If the hostname is returned as ending
 * in ":X" then trim the ":X" and use X and remote nic_id, else remote nic_id
 * is 0.
 */
int
myri_mac_to_hostname(
  int nic_id,
  lf_mac_addr_t mac_addr,
  lf_string_t hostname,
  int *his_nic_id)
{
  sprintf (hostname, LF_MAC_FORMAT, LF_MAC_ARGS(mac_addr));
  return 0;
}

/*
 * Return information about this NIC
 */
int
myri_get_nic_info(
  int nic_id,
  struct myri_nic_info *nip)
{
    char *mac;

    mac = getenv ("LXGDB_MAC");
    if (!mac)
      mac = "LXGDB_MAC=00-60-dd-46-4d-41";
    if (strlen (mac) != 17) {
      fprintf (stderr, "LXGDB_MAC is incorrectly to \"%s\".\n", mac);
      return 1;
    }
    
    /* get MAC address */
    nip->mac_addr[0] = strtol (mac+0, 0, 16);
    nip->mac_addr[1] = strtol (mac+3, 0, 16);
    nip->mac_addr[2] = strtol (mac+6, 0, 16);
    nip->mac_addr[3] = strtol (mac+9, 0, 16);
    nip->mac_addr[4] = strtol (mac+12, 0, 16);
    nip->mac_addr[5] = strtol (mac+15, 0, 16);

    nip->num_ports = 2;
    nip->num_routes = 1;
    sprintf (nip->serial_no, "serial no");
    sprintf (nip->product_id, "product id");

  return 0;
}

int myri_set_route_start(
  int nic_id)
{
  return 0;
}

int myri_set_route_end(
  int nic_id,
  lf_mac_addr_t mapper_mac,
  int map_version,
  int num_hosts)
{
  return 0;
}

/*
 * Set a route
 */
int
myri_set_route(int nic_id,
	       lf_mac_addr_t remote_mac,
	       enum lf_firmware_type fw_type,
	       int remote_port,
	       int route_num,
	       int local_port,
	       unsigned char *route,
	       int route_len)
{
  return 0;
}

/*
 * Return counters for a NIC port
 */
int
myri_get_nic_counters(
  int nic_id,
  int port,
  struct myri_nic_counters *counters)
{
  memset(counters, 0, sizeof(*counters));
  return 0;
}

/*
 * Return firmware code
 */
enum lf_firmware_type
myri_firmware_type()
{
  return MYRI_FW_LXGDB_1;
}

/*
 * set info for NIC auto-reply to scouts if supported
 */
int
myri_set_nic_reply_info(
  int nic_id,
  void *blob,
  int size)
{
  return 0;
}
